using System;
using System.Collections;

using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>2/27/2005</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///		Daylight savings type wrapped around corresponding enum to 
	///		provide reliable and easy to use type-name association.
	///	</summary>

	#endregion

	public class DaylightSavingsTypeDefinition : IRecognizableByID
	{
		/// <summary>
		/// Standard comparer used to compare daylight savings types by name.
		/// </summary>
		private class UseOrderComparer : BusinessObjectPropertyComparerBase
		{
			protected override int CompareProperties( object x, object y )
			{
				return Comparer.DefaultInvariant.Compare( (int)((DaylightSavingsTypeDefinition)x)._useOrder, (int)((DaylightSavingsTypeDefinition)y)._useOrder);
			}
		}

		private readonly DaylightSavingsType _type;
		private readonly string _name;
		private readonly int _useOrder;

		/// <summary>
		/// Constructs an instance of the class from the supplied type. 
		/// </summary>
		/// <param name="type">Daylight savings type.</param>
		protected DaylightSavingsTypeDefinition( DaylightSavingsType type )
		{
			_type = type;			
			_name = GetDaylightSavingsTypeString( type );
			_useOrder = GetUseOrder( type );
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6735"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>Valid type.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6736"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Daylight savings type (ID).
		/// </summary>
		public DaylightSavingsType Type
		{
			get
			{
				return _type;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6737"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>Valid DST name corresponding to type.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6738"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Daylight savings type name.
		/// </summary>
		public string Name 
		{
			get
			{
				return _name;
			}
		}	

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6741"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>Non-null non-empty list.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6743"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>List correctly sorted by frequency of use.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6742"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves all daylight savings types sorted by usage.
		/// </summary>
		/// <returns>A sorted hashlist containing all registered daylight savings types.</returns>
		public static BusinessObjectSortedHashList GetAllSortedByUse()
		{
			Array _allValues = Enum.GetValues( typeof(DaylightSavingsType) );
			BusinessObjectSortedHashList _list = new BusinessObjectSortedHashList( new UseOrderComparer(), _allValues.Length );

			foreach( DaylightSavingsType _type in _allValues )
				_list.Add( new DaylightSavingsTypeDefinition( _type ) );

			return _list;
		}
		
		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6739"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>Non-null ID (DST type).</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6740"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns unique object ID (daylight savings type).
		/// </summary>
		/// <returns>Unique object ID (daylight savings type).</returns>
		public object GetObjectID()
		{
			return Type;
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4660"> 
		///		<ExpectedInput>Valid DST ID.</ExpectedInput>
		///		<ExpectedOutput>Corresponding enum value.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4661"> 
		///		<ExpectedInput>Invalid DST ID.</ExpectedInput>
		///		<ExpectedOutput>ArgumentOutOfRangeException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Converts supplied integer ID to daylight savings type.
		/// </summary>
		/// <param name="dstId">Daylight savings type ID.</param>
		/// <returns>Appropriate value of <see cref="DaylightSavingsType"/> enum.</returns>
		public static DaylightSavingsType GetDaylightSavingsTypeFromId( int dstId )
		{
			switch( dstId )
			{
				case 1: 
					return DaylightSavingsType.DoNotObserve;
				case 2:
					return DaylightSavingsType.UsaStandard;
				case 3:
					return DaylightSavingsType.Custom;
				case 4:
					return DaylightSavingsType.Unknown;
				default:
					throw( new ArgumentOutOfRangeException( Common.StrRes.SysErrMsg.Common.UnableToConvertDSTCodeToEnum( dstId ).ResString ) );
			}
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4662"> 
		///		<ExpectedInput>Valid DST type.</ExpectedInput>
		///		<ExpectedOutput>Corresponding ID.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4663"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Converts supplied <see cref="DaylightSavingsType"/> value to integer ID.
		/// </summary>
		/// <param name="dst"><see cref="DaylightSavingsType"/> type to convert.</param>
		/// <returns>Integer ID corresponding to supplied <see cref="DaylightSavingsType"/>.</returns>
		public static int GetDaylightSavingsTypeId( DaylightSavingsType dst )
		{
			return (int)dst;
		}

		/// <summary>
		/// Gets daylight savings type name as a string.
		/// </summary>
		/// <param name="dst">Daylight savings type.</param>
		/// <returns>Name associated with the supplied daylight savings type.</returns>
		protected virtual string GetDaylightSavingsTypeString( DaylightSavingsType dst )
		{
			StrRes.StringResourceItem _resItem;

			switch( dst )
			{
				case DaylightSavingsType.DoNotObserve:
					_resItem = StrRes.OtherMsg.Common.DstTypeDoNotObserve();
					break;
				case DaylightSavingsType.UsaStandard:
					_resItem = StrRes.OtherMsg.Common.DstTypeUsaStandard();
					break;
				case DaylightSavingsType.Custom:
					_resItem = StrRes.OtherMsg.Common.DstTypeCustom();
					break;
				default:
					_resItem = StrRes.OtherMsg.Common.DstTypeUnknown();
					break;
			}

			return _resItem.ResString;
		}

		/// <summary>
		/// Gets use order associated with the DST.
		/// </summary>
		/// <param name="dst">Daylight savings type.</param>
		/// <returns>Int sort order.</returns>
		protected virtual int GetUseOrder( DaylightSavingsType dst )
		{
			switch( dst )
			{
				case DaylightSavingsType.DoNotObserve:
					return 1;
				case DaylightSavingsType.UsaStandard:
					return 0;
				case DaylightSavingsType.Custom:
					return 2;
				default:
					return 1000;
			}
		}
	}
}
